# 스트래티지 패턴 (Strategy Pattern)

## 스트래티지 패턴이란?
- **행위**을 클래스로 캡슐화해 동적으로 행위를 자유롭게 바꿀 수 있게 해주는 패턴
  - 같은 문제를 해결하는 여러 알고리즘이 클래스별로 캡술화되어 있어서 동일한 문제를 다른 알고리즘으로 해결하고 싶을 때 교체 가능
  - **행위 패턴**중 하나 
- **전략(strategy)** 를 쉽게 바꿀 수 있도록 하는 디자인 패턴이다.
> 전략 패턴이라고도 한다.

## 구조
<img width="700" src = "https://user-images.githubusercontent.com/102718303/212553610-86feb082-173a-4431-9b4d-f5018700cf92.png">

</br>

- `Strategy`
  - 인터페이스나 추상 클래스로 외부에서 동일한 방식으로 알고리즘을 호출하는 방법을 명시
- `ConcreateStrategy`
  - 여러 알고리즘을 실제로 구현한 클래스
- `Context`
  - 스트래지티 패턴을 이용하는 역할
  - 구체적인 전략을 바꿀 수 있는 `setter` 매소드 제공

## 예시
<img width="600" src = "https://user-images.githubusercontent.com/102718303/212553643-8cc87de7-f224-4812-bd9b-80699b4466f9.png">

</br>

```Java
public abstract class Robot {
  private String name;
  public Robot(String name) { this.name = name; }
  public String getName() { return name; }
  // 추상 메서드
  public abstract void attack();
  public abstract void move();
}

public class TaekwonV extends Robot {
  public TaekwonV(String name) { super(name); }
  public void attack() { System.out.println("I have Missile."); }
  public void move() { System.out.println("I can only walk."); }
}
public class Atom extends Robot {
  public Atom(String name) { super(name); }
  public void attack() { System.out.println("I have strong punch."); }
  public void move() { System.out.println("I can fly."); }
}
```
### 문제점
1. 기존 로봇의 공격과 이동 방법을 수정하는 경우
   - Atom이 못 날고 걷게한다면?
   - TaekwonV를 날게 하려면?
   - 새로운 기능을 위해 기존 코드의 내용을 수정해야 하므로 **OCP**에 위배
   - `move()`메소드 중복
2. 새로운 로봇을 추가해 기존의 공격 또는 이동방법을 추가/수정하는 경우
   - `move()` 혹은 `attack()`메소드가 중복

</br>


### 해결책
- 문제를 해결하기 위해서 **무엇이 변화**되는지 찾고, 클래스로 **캡슐화** 해야한다.
- 즉, **이동 방식**과 **공격 방식**이 변화되고 있기 때문에, 이를 위한 인터페이스를 구현하고 실제 실현한 클래스를 만든다.

</br>


<img width="800" src = "https://user-images.githubusercontent.com/102718303/212553666-3b1be183-625a-4e3e-832d-5171552a167b.png">

- `Robot`클래스가 이동 기능과 공격 기능을 이용하는 `Context`의 역할
- 양쪽의 `MovingStaratgy`와 `AttackStrategy`가 `Strategy`의 역할 (캡슐화)
- 스트래지티 패턴을 이용하면 새로운 기능의 추가가 기존 코드의 영향을 미치지 못하므로 **OCP**을 만족한다.

```Java
// **Robot 클래스** //
public abstract class Robot {
private String name;
private AttackStrategy attackStrategy;
private MovingStrategy movingStrategy;

public Robot(String name) { this.name = name; }
public String getName() { return name; }
public void attack() { attackStrategy.attack(); }
public void move() { movingStrategy.move(); }

// setter 메서드
public void setAttackStrategy(AttackStrategy attackStrategy) {
  this.attackStrategy = attackStrategy; }
public void setMovingStrategy(MovingStrategy movingStrategy) {
  this.movingStrategy = movingStrategy; }
}

// **공격 인터페이스** //
interface AttackStrategy { public void attack(); }
// 구체적인 클래스
public class MissileStrategy implements AttackStrategy {
  public void attack() { System.out.println("I have Missile."); }
}
public class PunchStrategy implements AttackStrategy {
  public void attack() { System.out.println("I have strong punch."); }
}

// **이동 인터페이스** ??
interface MovingStrategy { public void move(); }
// 구체적인 클래스
public class FlyingStrategy implements MovingStrategy {
  public void move() { System.out.println("I can fly."); }
}
public class WalkingStrategy implements MovingStrategy {
  public void move() { System.out.println("I can only walk."); }
}

// **실제 사용** // 
public class Client {
public static void main(String[] args) {
  Robot taekwonV = new TaekwonV("TaekwonV");
  Robot atom = new Atom("Atom");

  /* 수정된 부분: 전략 변경 방법 */
  taekwonV.setMovingStrategy(new WalkingStrategy());
  taekwonV.setAttackStrategy(new MissileStrategy());
  atom.setMovingStrategy(new FlyingStrategy());
  atom.setAttackStrategy(new PunchStrategy());
  
}
```

----
## Referance
- https://gmlwjd9405.github.io/2018/07/06/strategy-pattern.html




